// Emergency Response JavaScript

document.addEventListener('DOMContentLoaded', function() {
    initializeEmergencyForm();
    initializeSymptomChecker();
    setupFormValidation();
    initializeModalHandlers();
});

// Initialize emergency form
function initializeEmergencyForm() {
    const incidentTypeSelect = document.getElementById('incidentType');
    const symptomsGrid = document.getElementById('symptomsGrid');
    
    if (incidentTypeSelect && symptomsGrid) {
        incidentTypeSelect.addEventListener('change', function() {
            updateSymptomsGrid(this.value);
        });
    }
    
    // Form submission handler
    const emergencyForm = document.getElementById('emergencyForm');
    if (emergencyForm) {
        emergencyForm.addEventListener('submit', handleEmergencySubmit);
    }
}

// Update symptoms grid based on incident type
function updateSymptomsGrid(incidentType) {
    const symptomsGrid = document.getElementById('symptomsGrid');
    
    if (!incidentType || !window.emergencyTypes || !window.emergencyTypes[incidentType]) {
        symptomsGrid.innerHTML = '<p>Please select an incident type to see relevant symptoms.</p>';
        return;
    }
    
    const symptoms = window.emergencyTypes[incidentType].symptoms;
    symptomsGrid.innerHTML = '';
    
    symptoms.forEach((symptom, index) => {
        const checkboxDiv = document.createElement('div');
        checkboxDiv.className = 'symptom-checkbox';
        checkboxDiv.innerHTML = `
            <input type="checkbox" id="symptom_${index}" name="symptoms[]" value="${symptom}">
            <label for="symptom_${index}">${symptom}</label>
        `;
        
        // Add click handler for visual feedback
        const checkbox = checkboxDiv.querySelector('input[type="checkbox"]');
        checkbox.addEventListener('change', function() {
            if (this.checked) {
                checkboxDiv.classList.add('checked');
            } else {
                checkboxDiv.classList.remove('checked');
            }
        });
        
        symptomsGrid.appendChild(checkboxDiv);
    });
}

// Initialize symptom checker functionality
function initializeSymptomChecker() {
    // Auto-suggest severity based on selected symptoms
    document.addEventListener('change', function(e) {
        if (e.target.name === 'symptoms[]') {
            updateSeveritySuggestion();
        }
    });
}

// Update severity suggestion based on symptoms
function updateSeveritySuggestion() {
    const checkedSymptoms = document.querySelectorAll('input[name="symptoms[]"]:checked');
    const severitySelect = document.getElementById('severity');
    
    if (checkedSymptoms.length === 0) return;
    
    const criticalSymptoms = ['Loss of consciousness', 'High body temperature', 'Confusion', 'Hot, dry skin'];
    const highSymptoms = ['Rapid pulse', 'Extreme thirst', 'Heavy sweating'];
    
    let suggestedSeverity = 'low';
    
    Array.from(checkedSymptoms).forEach(checkbox => {
        if (criticalSymptoms.includes(checkbox.value)) {
            suggestedSeverity = 'critical';
        } else if (highSymptoms.includes(checkbox.value) && suggestedSeverity !== 'critical') {
            suggestedSeverity = 'high';
        } else if (suggestedSeverity === 'low') {
            suggestedSeverity = 'medium';
        }
    });
    
    // Highlight suggested severity
    Array.from(severitySelect.options).forEach(option => {
        if (option.value === suggestedSeverity) {
            option.style.backgroundColor = '#fef3c7';
            option.style.fontWeight = 'bold';
        } else {
            option.style.backgroundColor = '';
            option.style.fontWeight = '';
        }
    });
}

