// Profile Settings JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // Initialize profile settings
    initializeProfileSettings();
});

function initializeProfileSettings() {
    // Add event listeners
    setupEventListeners();
    
    // Initialize components
    initializeSidebar();
    initializeTabs();
    initializePasswordStrength();
    initializeFormValidation();
    
    console.log('Profile settings initialized');
}

function setupEventListeners() {
    // Menu toggle for mobile
    const menuToggle = document.getElementById('menuToggle');
    if (menuToggle) {
        menuToggle.addEventListener('click', toggleSidebar);
    }

    // Tab navigation
    const tabButtons = document.querySelectorAll('.tab-btn');
    tabButtons.forEach(button => {
        button.addEventListener('click', function() {
            switchTab(this.dataset.tab);
        });
    });

    // Form submissions
    setupFormSubmissions();
    
    // Avatar upload
    const avatarUploadBtn = document.querySelector('.avatar-upload-btn');
    if (avatarUploadBtn) {
        avatarUploadBtn.addEventListener('click', handleAvatarUpload);
    }

    // Toggle switches
    setupToggleSwitches();
    
    // Password strength monitoring
    const newPasswordInput = document.getElementById('newPassword');
    if (newPasswordInput) {
        newPasswordInput.addEventListener('input', updatePasswordStrength);
    }

    // Emergency contact management
    setupEmergencyContactHandlers();
    
    // Session management
    setupSessionHandlers();
}

function initializeSidebar() {
    const sidebar = document.getElementById('sidebar');
    const menuToggle = document.getElementById('menuToggle');

    // Close sidebar when clicking outside on mobile
    document.addEventListener('click', function(e) {
        if (window.innerWidth <= 1024) {
            if (!sidebar.contains(e.target) && !menuToggle.contains(e.target)) {
                sidebar.classList.remove('open');
            }
        }
    });
}

function toggleSidebar() {
    const sidebar = document.getElementById('sidebar');
    sidebar.classList.toggle('open');
}

function initializeTabs() {
    // Set default active tab
    const defaultTab = 'personal';
    switchTab(defaultTab);
}

function switchTab(tabName) {
    // Remove active class from all tabs and contents
    document.querySelectorAll('.tab-btn').forEach(btn => btn.classList.remove('active'));
    document.querySelectorAll('.tab-content').forEach(content => content.classList.remove('active'));
    
    // Add active class to selected tab and content
    document.querySelector(`[data-tab="${tabName}"]`).classList.add('active');
    document.getElementById(tabName).classList.add('active');
    
    console.log(`Switched to ${tabName} tab`);
}

function setupFormSubmissions() {
    // Personal information form
    const personalForm = document.getElementById('personalForm');
    if (personalForm) {
        personalForm.addEventListener('submit', handlePersonalFormSubmit);
    }

    // Security form
    const securityForm = document.getElementById('securityForm');
    if (securityForm) {
        securityForm.addEventListener('submit', handleSecurityFormSubmit);
    }

    // Notifications form
    const notificationsForm = document.getElementById('notificationsForm');
    if (notificationsForm) {
        notificationsForm.addEventListener('submit', handleNotificationsFormSubmit);
    }
}

function handlePersonalFormSubmit(e) {
    e.preventDefault();
    
    showLoadingState();
    
    // Get form data
    const formData = new FormData(e.target);
    const data = Object.fromEntries(formData);
    
    console.log('Updating personal information:', data);
    
    // Simulate API call
    setTimeout(() => {
        hideLoadingState();
        showNotification('Personal information updated successfully!', 'success');
        
        // Update profile header if name changed
        if (data.fullName) {
            updateProfileHeader(data);
        }
    }, 2000);
}

function handleSecurityFormSubmit(e) {
    e.preventDefault();
    
    const currentPassword = document.getElementById('currentPassword').value;
    const newPassword = document.getElementById('newPassword').value;
    const confirmPassword = document.getElementById('confirmPassword').value;
    
    // Validate passwords
    if (!currentPassword || !newPassword || !confirmPassword) {
        showNotification('Please fill in all password fields', 'error');
        return;
    }
    
    if (newPassword !== confirmPassword) {
        showNotification('New passwords do not match', 'error');
        return;
    }
    
    if (newPassword.length < 8) {
        showNotification('Password must be at least 8 characters long', 'error');
        return;
    }
    
    showLoadingState();
    
    console.log('Updating password...');
    
    // Simulate API call
    setTimeout(() => {
        hideLoadingState();
        showNotification('Password updated successfully!', 'success');
        
        // Clear password fields
        e.target.reset();
        updatePasswordStrength({ target: { value: '' } });
    }, 2000);
}

