// Reporting System JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // Initialize reporting system
    initializeReportingSystem();
});

function initializeReportingSystem() {
    // Add event listeners
    setupEventListeners();
    
    // Initialize components
    initializeSidebar();
    initializeFilters();
    
    console.log('Reporting System initialized');
}

function setupEventListeners() {
    // Menu toggle for mobile
    const menuToggle = document.getElementById('menuToggle');
    if (menuToggle) {
        menuToggle.addEventListener('click', toggleSidebar);
    }

    // Filter functionality
    const filterSelect = document.querySelector('.filter-select');
    if (filterSelect) {
        filterSelect.addEventListener('change', filterReports);
    }

    const filterBtn = document.querySelector('.filter-btn');
    if (filterBtn) {
        filterBtn.addEventListener('click', applyFilters);
    }

    // Export buttons
    setupExportButtons();
    
    // Action buttons in table
    setupTableActions();
}

function initializeSidebar() {
    const sidebar = document.getElementById('sidebar');
    const menuToggle = document.getElementById('menuToggle');

    // Close sidebar when clicking outside on mobile
    document.addEventListener('click', function(e) {
        if (window.innerWidth <= 1024) {
            if (!sidebar.contains(e.target) && !menuToggle.contains(e.target)) {
                sidebar.classList.remove('open');
            }
        }
    });
}

function toggleSidebar() {
    const sidebar = document.getElementById('sidebar');
    sidebar.classList.toggle('open');
}

function initializeFilters() {
    // Set default filter state
    const filterSelect = document.querySelector('.filter-select');
    if (filterSelect) {
        filterSelect.value = 'all';
    }
}

function generateReport(reportType) {
    // Show loading state
    showLoadingState();
    
    console.log(`Generating ${reportType} report...`);
    
    // Simulate report generation
    setTimeout(() => {
        hideLoadingState();
        showNotification(`${formatReportType(reportType)} is being generated. You will be notified when it's ready.`, 'success');
        
        // Simulate adding to recent reports (in real app, this would be handled by backend)
        addToRecentReports(reportType);
    }, 2000);
}

function formatReportType(reportType) {
    return reportType.split('-').map(word => 
        word.charAt(0).toUpperCase() + word.slice(1)
    ).join(' ') + ' Report';
}

function addToRecentReports(reportType) {
    // In a real application, this would update the database
    // For demo purposes, we'll just show a notification
    console.log(`Added ${reportType} report to recent reports`);
}

function filterReports() {
    const filterValue = document.querySelector('.filter-select').value;
    const tableRows = document.querySelectorAll('.table-row');
    
    tableRows.forEach(row => {
        const reportType = row.querySelector('.table-col:nth-child(2)').textContent.toLowerCase();
        
        if (filterValue === 'all' || reportType.includes(filterValue.replace('-', ' '))) {
            row.style.display = 'grid';
            row.classList.add('fade-in');
        } else {
            row.style.display = 'none';
        }
    });
    
    showNotification(`Filtered reports by: ${filterValue === 'all' ? 'All Reports' : formatReportType(filterValue)}`, 'info');
}

function applyFilters() {
    // Advanced filtering logic would go here
    filterReports();
}

function setupExportButtons() {
    const exportButtons = document.querySelectorAll('.export-btn');
    
    exportButtons.forEach(button => {
        button.addEventListener('click', function() {
            const exportType = this.classList[1].replace('-export', '');
            handleExport(exportType);
        });
    });
}

function handleExport(exportType) {
    showLoadingState();
    
    console.log(`Exporting reports as ${exportType.toUpperCase()}...`);
    
    // Simulate export process
    setTimeout(() => {
        hideLoadingState();
        showNotification(`Reports exported as ${exportType.toUpperCase()} successfully!`, 'success');
    }, 1500);
}

function setupTableActions() {
    // View buttons
    const viewButtons = document.querySelectorAll('.view-btn');
    viewButtons.forEach(button => {
        button.addEventListener('click', function() {
            const reportId = this.closest('.table-row').querySelector('.report-id').textContent;
            viewReport(reportId);
        });
    });

    // Download buttons
    const downloadButtons = document.querySelectorAll('.download-btn');
    downloadButtons.forEach(button => {
        button.addEventListener('click', function() {
            const reportId = this.closest('.table-row').querySelector('.report-id').textContent;
            downloadReport(reportId);
        });
    });

    // Share buttons
    const shareButtons = document.querySelectorAll('.share-btn');
    shareButtons.forEach(button => {
        button.addEventListener('click', function() {
            const reportId = this.closest('.table-row').querySelector('.report-id').textContent;
            shareReport(reportId);
        });
    });
}

function viewReport(reportId) {
    showLoadingState();
    
    console.log(`Viewing report: ${reportId}`);
    
    // Simulate loading report
    setTimeout(() => {
        hideLoadingState();
        showNotification(`Opening report ${reportId}...`, 'info');
        // In real app, this would open a modal or navigate to report details
    }, 1000);
}

function downloadReport(reportId) {
    showLoadingState();
    
    console.log(`Downloading report: ${reportId}`);
    
    // Simulate download
    setTimeout(() => {
        hideLoadingState();
        showNotification(`Report ${reportId} downloaded successfully!`, 'success');
    }, 1500);
}

function shareReport(reportId) {
    console.log(`Sharing report: ${reportId}`);
    
    // Simulate share functionality
    if (navigator.share) {
        navigator.share({
            title: `HeatGuard Report ${reportId}`,
            text: 'Heat safety report from HeatGuard system',
            url: window.location.href
        }).catch(console.error);
    } else {
        // Fallback for browsers that don't support Web Share API
        copyToClipboard(window.location.href + '#' + reportId);
        showNotification('Report link copied to clipboard!', 'success');
    }
}

function copyToClipboard(text) {
    if (navigator.clipboard) {
        navigator.clipboard.writeText(text);
    } else {
        // Fallback for older browsers
        const textArea = document.createElement('textarea');
        textArea.value = text;
        document.body.appendChild(textArea);
        textArea.select();
        document.execCommand('copy');
        document.body.removeChild(textArea);
    }
}

function showLoadingState() {
    const reportsContent = document.querySelector('.reports-content');
    if (reportsContent) {
        reportsContent.classList.add('loading');
    }
}

function hideLoadingState() {
    const reportsContent = document.querySelector('.reports-content');
    if (reportsContent) {
        reportsContent.classList.remove('loading');
    }
}

function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <i class="fas fa-${getNotificationIcon(type)}"></i>
            <span>${message}</span>
        </div>
        <button class="notification-close">&times;</button>
    `;
    
    // Add styles
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${getNotificationColor(type)};
        color: white;
        padding: 16px 20px;
        border-radius: 12px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2);
        z-index: 10000;
        display: flex;
        align-items: center;
        justify-content: space-between;
        min-width: 300px;
        animation: slideInRight 0.3s ease;
    `;
    
    // Add notification to DOM
    document.body.appendChild(notification);
    
    // Close button functionality
    notification.querySelector('.notification-close').addEventListener('click', () => {
        notification.remove();
    });
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.remove();
        }
    }, 5000);
}

function getNotificationIcon(type) {
    switch (type) {
        case 'success': return 'check-circle';
        case 'error': return 'exclamation-circle';
        case 'warning': return 'exclamation-triangle';
        default: return 'info-circle';
    }
}

function getNotificationColor(type) {
    switch (type) {
        case 'success': return '#10b981';
        case 'error': return '#ef4444';
        case 'warning': return '#f59e0b';
        default: return '#3b82f6';
    }
}

// Add CSS for notification animation
const style = document.createElement('style');
style.textContent = `
    @keyframes slideInRight {
        from {
            transform: translateX(100%);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
    
    .notification-content {
        display: flex;
        align-items: center;
        gap: 10px;
    }
    
    .notification-close {
        background: none;
        border: none;
        color: white;
        font-size: 18px;
        cursor: pointer;
        padding: 0;
        margin-left: 15px;
    }
    
    .notification-close:hover {
        opacity: 0.7;
    }
`;
document.head.appendChild(style);

// Handle window resize
window.addEventListener('resize', function() {
    const sidebar = document.getElementById('sidebar');
    if (window.innerWidth > 1024) {
        sidebar.classList.remove('open');
    }
});

// Refresh data periodically (simulate real-time updates)
setInterval(function() {
    // In a real application, this would fetch updated report data
    console.log('Checking for new reports...');
}, 30000); // Check every 30 seconds