// Safety Guidelines JavaScript

document.addEventListener('DOMContentLoaded', function() {
    initializeTabs();
    initializeQuickActions();
});

// Tab functionality
function initializeTabs() {
    const tabButtons = document.querySelectorAll('.tab-btn');
    const tabContents = document.querySelectorAll('.tab-content');
    
    tabButtons.forEach(button => {
        button.addEventListener('click', function() {
            const tabId = this.dataset.tab;
            
            // Remove active class from all tabs and content
            tabButtons.forEach(btn => btn.classList.remove('active'));
            tabContents.forEach(content => content.classList.remove('active'));
            
            // Add active class to clicked tab and corresponding content
            this.classList.add('active');
            document.getElementById(tabId + '-content').classList.add('active');
        });
    });
}

// Show current guidelines based on heat level
function showCurrentGuidelines() {
    const currentHeatLevel = document.querySelector('.heat-status').textContent.toLowerCase();
    const tabButton = document.querySelector(`[data-tab="${currentHeatLevel}"]`);
    
    if (tabButton) {
        tabButton.click();
        // Smooth scroll to guidelines
        document.querySelector('.guidelines-tabs').scrollIntoView({
            behavior: 'smooth',
            block: 'start'
        });
    }
}

// Emergency contact functions
function callEmergency(number) {
    // In a real app, this would integrate with device calling capabilities
    if (confirm(`Call ${number}?`)) {
        // For demo purposes, just show alert
        alert(`Calling ${number}...`);
        // In production: window.location.href = `tel:${number}`;
    }
}

// Download guidelines as PDF
function downloadGuidelines() {
    // Simulate PDF download
    const link = document.createElement('a');
    link.href = '#'; // In production, this would be the actual PDF URL
    link.download = 'HeatGuard_Safety_Guidelines.pdf';
    
    // Show loading state
    const downloadBtn = document.querySelector('.btn-download');
    const originalText = downloadBtn.innerHTML;
    downloadBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Generating PDF...';
    downloadBtn.disabled = true;
    
    setTimeout(() => {
        downloadBtn.innerHTML = originalText;
        downloadBtn.disabled = false;
        alert('Safety Guidelines PDF downloaded successfully!');
    }, 2000);
}

// Print guidelines
function printGuidelines() {
    // Show all tab contents for printing
    const tabContents = document.querySelectorAll('.tab-content');
    tabContents.forEach(content => {
        content.style.display = 'block';
    });
    
    window.print();
    
    // Restore original display after printing
    setTimeout(() => {
        tabContents.forEach((content, index) => {
            if (!content.classList.contains('active')) {
                content.style.display = 'none';
            }
        });
    }, 1000);
}

// Initialize quick actions
function initializeQuickActions() {
    // Add click tracking for analytics
    const actionButtons = document.querySelectorAll('.action-btn');
    actionButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            const action = this.textContent.trim();
            console.log(`Safety Guidelines Action: ${action}`);
            
            // Add visual feedback
            this.style.transform = 'scale(0.95)';
            setTimeout(() => {
                this.style.transform = '';
            }, 150);
        });
    });
}

// Auto-refresh current heat index (simulation)
function updateHeatIndex() {
    const heatValue = document.querySelector('.heat-value');
    const heatStatus = document.querySelector('.heat-status');
    
    // Simulate API call to get current temperature
    // In production, this would be an actual API call
    fetch('/api/current-heat-index')
        .then(response => response.json())
        .then(data => {
            heatValue.textContent = data.temperature + '°C';
            heatStatus.textContent = data.level;
            heatStatus.className = `heat-status status-${data.level.toLowerCase()}`;
        })
        .catch(error => {
            console.log('Heat index update simulated');
        });
}

// Update heat index every 5 minutes
setInterval(updateHeatIndex, 300000);

// Keyboard shortcuts
document.addEventListener('keydown', function(e) {
    // Ctrl + P for print
    if (e.ctrlKey && e.key === 'p') {
        e.preventDefault();
        printGuidelines();
    }
    
    // Ctrl + D for download
    if (e.ctrlKey && e.key === 'd') {
        e.preventDefault();
        downloadGuidelines();
    }
    
    // Ctrl + E for emergency
    if (e.ctrlKey && e.key === 'e') {
        e.preventDefault();
        window.location.href = 'emergency-response.php';
    }
});

// Add fade-in animation to cards
function addFadeInAnimation() {
    const cards = document.querySelectorAll('.guidelines-card, .contact-card');
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.style.opacity = '1';
                entry.target.style.transform = 'translateY(0)';
            }
        });
    });
    
    cards.forEach(card => {
        card.style.opacity = '0';
        card.style.transform = 'translateY(20px)';
        card.style.transition = 'opacity 0.5s ease, transform 0.5s ease';
        observer.observe(card);
    });
}

// Initialize animations when page loads
window.addEventListener('load', addFadeInAnimation);