
document.addEventListener('DOMContentLoaded', function() {
    initializeScheduleManagement();
    setupEventListeners();
    startScheduleMonitoring();
});

// Global variables
let scheduleUpdateInterval;
let officers = [];
let currentHeatIndex = 42;
let alerts = [];

// Initialize schedule management system
function initializeScheduleManagement() {
    console.log('Schedule Management System initialized');
    loadOfficers();
    updateScheduleRecommendations();
    setupModalEvents();
}

// Setup event listeners
function setupEventListeners() {
    // Menu toggle functionality
    const menuToggle = document.getElementById('menuToggle');
    const sidebar = document.getElementById('sidebar');
    
    if (menuToggle && sidebar) {
        menuToggle.addEventListener('click', function() {
            sidebar.classList.toggle('open');
        });
    }

    // Officer item hover effects
    const officerItems = document.querySelectorAll('.officer-item');
    officerItems.forEach(item => {
        item.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-2px)';
        });
        
        item.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    });

    // Alert item hover effects
    const alertItems = document.querySelectorAll('.alert-item');
    alertItems.forEach(item => {
        item.addEventListener('mouseenter', function() {
            this.style.background = '#f1f5f9';
        });
        
        item.addEventListener('mouseleave', function() {
            this.style.background = '#f8fafc';
        });
    });
}

// Load officers data
function loadOfficers() {
    // In a real application, this would fetch from a database
    officers = [
        {
            id: 1,
            name: 'Officer Jerome Buntalidad',
            badge: 'TMU-2024-001',
            position: 'City Hall Intersection',
            status: 'on_duty',
            exposureTime: 180,
            shiftStart: '06:00',
            shiftEnd: '14:00'
        },
        {
            id: 2,
            name: 'Officer Maria Santos',
            badge: 'TMU-2024-002',
            position: 'Public Plaza',
            status: 'on_break',
            exposureTime: 160,
            shiftStart: '06:00',
            shiftEnd: '14:00'
        }
    ];
}

// Start schedule monitoring
function startScheduleMonitoring() {
    scheduleUpdateInterval = setInterval(() => {
        updateExposureTimes();
        checkForAlerts();
        updateScheduleDisplay();
    }, 60000); // Update every minute
}

// Update exposure times for active officers
function updateExposureTimes() {
    officers.forEach(officer => {
        if (officer.status === 'on_duty') {
            officer.exposureTime += 1; // Add 1 minute
            updateOfficerDisplay(officer);
        }
    });
}



// Get recommended break interval based on heat index
function getBreakInterval(heatIndex) {
    if (heatIndex >= 40) return 60; // 1 hour
    if (heatIndex >= 32) return 90; // 1.5 hours
    return 120; // 2 hours
}

// Add new officer
function addNewOfficer() {
    const modal = document.getElementById('addOfficerModal');
    modal.classList.add('show');
    modal.style.display = 'flex';
}

// Close modal
function closeModal(modalId) {
    const modal = document.getElementById(modalId);
    modal.classList.remove('show');
    setTimeout(() => {
        modal.style.display = 'none';
    }, 300);
}

// Setup modal events
function setupModalEvents() {
    // Close modal when clicking outside
    const modals = document.querySelectorAll('.modal');
    modals.forEach(modal => {
        modal.addEventListener('click', function(e) {
            if (e.target === this) {
                closeModal(this.id);
            }
        });
    });
}





// Update schedule recommendations
function updateScheduleRecommendations() {
    const recommendationsElement = document.getElementById('scheduleRecommendations');
    if (!recommendationsElement) return;
    
    const breakInterval = getBreakInterval(currentHeatIndex);
    const activeOfficers = officers.filter(o => o.status === 'on_duty').length;
    const officersOnBreak = officers.filter(o => o.status === 'on_break').length;
    
    let recommendations = [];
    
    // Heat-based recommendations
    if (currentHeatIndex >= 40) {
        recommendations.push({
            type: 'warning',
            message: `Extreme heat conditions (${currentHeatIndex}°C). Recommend ${breakInterval}-minute break intervals.`
        });
    } else if (currentHeatIndex >= 32) {
        recommendations.push({
            type: 'caution',
            message: `High heat conditions (${currentHeatIndex}°C). Monitor officer exposure closely.`
        });
    }
    
    // Staffing recommendations
    if (activeOfficers < 2) {
        recommendations.push({
            type: 'info',
            message: 'Consider scheduling additional officers for adequate coverage.'
        });
    }
    
    if (officersOnBreak > activeOfficers) {
        recommendations.push({
            type: 'warning',
            message: 'More officers on break than active. Review break schedules.'
        });
    }
    
    // Display recommendations
    recommendationsElement.innerHTML = recommendations.length > 0 
        ? recommendations.map(rec => `
            <div class="recommendation-item ${rec.type}">
                <i class="fas ${getRecommendationIcon(rec.type)}"></i>
                <span>${rec.message}</span>
            </div>
        `).join('')
        : '<div class="recommendation-item info"><i class="fas fa-check-circle"></i><span>All officers are scheduled optimally for current conditions.</span></div>';
}


// Display alert
function displayAlert(alert) {
    const alertsContainer = document.getElementById('alertsContainer');
    if (!alertsContainer) return;
    
    const alertElement = document.createElement('div');
    alertElement.className = `alert-item priority-${alert.priority}`;
    alertElement.setAttribute('data-alert-id', alert.id);
    
    alertElement.innerHTML = `
        <div class="alert-icon">
            <i class="fas ${getAlertIcon(alert.priority)}"></i>
        </div>
        <div class="alert-content">
            <div class="alert-message">${alert.message}</div>
            <div class="alert-officer">${alert.officer}</div>
            <div class="alert-time">${formatTime(alert.timestamp)}</div>
        </div>
        <div class="alert-actions">
            <button class="btn-dismiss" onclick="dismissAlert(${alert.id})">
                <i class="fas fa-times"></i>
            </button>
        </div>
    `;
    
    alertsContainer.insertBefore(alertElement, alertsContainer.firstChild);
}

// Format time helper
function formatTime(date) {
    return date.toLocaleTimeString('en-US', { 
        hour: '2-digit', 
        minute: '2-digit' 
    });
}

// Cleanup function
function cleanup() {
    if (scheduleUpdateInterval) {
        clearInterval(scheduleUpdateInterval);
    }
}

// Initialize cleanup on page unload
window.addEventListener('beforeunload', cleanup);

// Wait for the page to load
document.addEventListener('DOMContentLoaded', function() {
    
    // Get the hamburger button and sidebar
    const menuButton = document.getElementById('menuToggle');
    const sidebar = document.getElementById('sidebar');
    
    // When hamburger button is clicked, show/hide sidebar
    menuButton.addEventListener('click', function() {
        sidebar.classList.toggle('open');
    });
    
    // Close sidebar when clicking outside of it (only on mobile)
    document.addEventListener('click', function(event) {
        // Check if click is outside sidebar and not on menu button
        const clickedInsideSidebar = sidebar.contains(event.target);
        const clickedMenuButton = menuButton.contains(event.target);
        
        if (!clickedInsideSidebar && !clickedMenuButton && sidebar.classList.contains('open')) {
            sidebar.classList.remove('open');
        }
    });
    
    // Emergency alert function (for the emergency button in your HTML)
    function triggerEmergency() {
        alert('Emergency Alert Triggered!');
    }
    
    // Make the function available globally so HTML can use it
    window.triggerEmergency = triggerEmergency;
    
});