// User Management JavaScript
document.addEventListener('DOMContentLoaded', function() {
    initializeUserManagement();
});

// Global variables
let currentUsers = window.userData || [];
let editingUserId = null;

function initializeUserManagement() {
    // Initialize search functionality
    const searchInput = document.getElementById('searchInput');
    if (searchInput) {
        searchInput.addEventListener('input', handleSearch);
    }

    // Initialize role description updates
    const roleSelect = document.getElementById('userRole');
    if (roleSelect) {
        roleSelect.addEventListener('change', updateRoleDescription);
    }

    // Initialize form submission
    const userForm = document.getElementById('userForm');
    if (userForm) {
        userForm.addEventListener('submit', handleFormSubmit);
    }

    // Initialize modal close on outside click
    window.addEventListener('click', function(event) {
        const modals = document.querySelectorAll('.modal');
        modals.forEach(modal => {
            if (event.target === modal) {
                modal.style.display = 'none';
            }
        });
    });
}

// Search functionality
function handleSearch(event) {
    const searchTerm = event.target.value.toLowerCase();
    const tableRows = document.querySelectorAll('#usersTableBody tr');
    
    tableRows.forEach(row => {
        const userName = row.querySelector('.user-details h4').textContent.toLowerCase();
        const userEmail = row.querySelector('.user-details p').textContent.toLowerCase();
        const userRole = row.querySelector('.role-badge').textContent.toLowerCase();
        const userDepartment = row.cells[2].textContent.toLowerCase();
        
        const matches = userName.includes(searchTerm) || 
                       userEmail.includes(searchTerm) || 
                       userRole.includes(searchTerm) || 
                       userDepartment.includes(searchTerm);
        
        row.style.display = matches ? '' : 'none';
    });
}

// Modal functions
function openAddUserModal() {
    editingUserId = null;
    document.getElementById('modalTitle').textContent = 'Add New User';
    document.getElementById('submitBtn').textContent = 'Add User';
    document.getElementById('userForm').reset();
    document.getElementById('userId').value = '';
    document.getElementById('userModal').style.display = 'block';
    updateRoleDescription();
}

// Form handling
function handleFormSubmit(event) {
    event.preventDefault();
    
    const formData = new FormData(event.target);
    const userData = {
        name: formData.get('fullName'),
        email: formData.get('email'),
        phone: formData.get('phone'),
        badge: formData.get('badgeNumber'),
        role: formData.get('userRole'),
        department: formData.get('department'),
        status: formData.get('userStatus'),
        password: formData.get('password')
    };

    // Validation
    if (!userData.name || !userData.email || !userData.badge || !userData.role || !userData.department) {
        showAlert('Please fill in all required fields.', 'error');
        return;
    }

    if (!isValidEmail(userData.email)) {
        showAlert('Please enter a valid email address.', 'error');
        return;
    }

    if (editingUserId) {
        updateUser(editingUserId, userData);
    } else {
        addUser(userData);
    }
}

// Export functionality
function exportUsers() {
    const csvContent = "data:text/csv;charset=utf-8," 
        + "Name,Email,Role,Department,Badge,Status,Created Date\n"
        + currentUsers.map(user => 
            `"${user.name}","${user.email}","${user.role}","${user.department}","${user.badge}","${user.status}","${user.created_at}"`
        ).join("\n");
    
    const encodedUri = encodeURI(csvContent);
    const link = document.createElement("a");
    link.setAttribute("href", encodedUri);
    link.setAttribute("download", "users_export.csv");
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    
    showAlert('User data exported successfully!', 'success');
}

